/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

		Back end functions called from front end
		
        $Id: pgpKeyBack.c,v 1.51.2.3 2001/04/23 22:31:16 ajivsov Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"
#include <string.h>

#include "pgpDebug.h"
#include "pgpKeyPriv.h"
#include "pgpErrors.h"
#include "pgpPassCach.h"
#include "pgpPubKey.h"
#include "pgpPktByte.h"
#include "pgpSigSpec.h"
#include "pgpKeySpec.h"
#include "pgpMem.h"
#include "pgpContext.h"
#include "pgpEnv.h"
#include "pgpTrustPriv.h"
#include "pgpX509Priv.h"
#include "pgpHashPriv.h"
#include "pgpRegExp.h"
#include "pgpRandomPoolPriv.h"
#include "pgpRndSeed.h"

#ifndef NULL
#define NULL 0
#endif

extern PGPByte sP11Module[255];

	PGPError
pgpFetchObjectData_back( PGPContextRef context, PGPUInt32 id,
	PGPByte **bufptr, PGPSize *buflen )
{
	PGPKeyDBObj *obj;
	PGPByte const *data;
	PGPByte *buf;
	PGPSize len;

	if (pgpRPCEnabled())
		return pgpFetchObjectData_backRPC(context, id, bufptr, buflen);
	obj = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( obj ) );
	data = pgpFetchObject( obj, &len );
	buf = pgpContextMemAlloc( context, len, 0 );
	if( IsNull( buf ) )
		return kPGPError_OutOfMemory;
	pgpCopyMemory( data, buf, len );
	*bufptr = buf;
	*buflen = len;
	return kPGPError_NoErr;
}


	PGPError 
pgpGetKeyByKeyID_back(
	PGPContextRef			context,
	PGPUInt32				dbid,
	PGPKeyID const *		keyIDIn,
	PGPBoolean				dummyOK,
	PGPBoolean				deletedOK,
	PGPUInt32 *				outID )
{
	PGPKeyDB *db;
	PGPKeyDBObj *obj;
	PGPError err;

	if (pgpRPCEnabled())
		return pgpGetKeyByKeyID_backRPC(context, dbid, keyIDIn, dummyOK, deletedOK, outID);
	*outID = 0;
	db = (PGPKeyDB *)dbid;
	pgpAssert( pgpKeyDBIsValid( db ) );
	err = pgpGetKeyByKeyID_internal( db, keyIDIn, dummyOK, deletedOK, &obj );
	if( IsPGPError( err ) )
		return err;
	*outID = (PGPUInt32) obj;
	return kPGPError_NoErr;
}

	PGPError
pgpKeyEncrypt_back( PGPContextRef context, PGPUInt32 id, PGPByte const *inbuf,
	PGPSize inbuflen, PGPPublicKeyMessageFormat format,
	PGPByte **outbuf, PGPSize *outbuflen )
{
	PGPKeyDBObj *	key;

	if (pgpRPCEnabled())
		return pgpKeyEncrypt_backRPC(context, id, inbuf, inbuflen, format, outbuf, outbuflen);
	key = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpKeyEncrypt_internal( key, inbuf, inbuflen, format,
								   outbuf, outbuflen );
}

	PGPError
pgpKeyDecrypt_back( PGPContextRef context, PGPUInt32 id,
	PGPByte const *passphrase, PGPSize pplen, PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	PGPByte const *inbuf, PGPSize inbuflen, PGPPublicKeyMessageFormat format,
	PGPByte **outbuf, PGPSize *outbuflen )
{
	PGPKeyDBObj *	key;

	if (pgpRPCEnabled())
		return pgpKeyDecrypt_backRPC(context, id, passphrase, pplen,
								hashedPhrase, cacheTimeOut, cacheGlobal,
								inbuf, inbuflen, format, outbuf, outbuflen);
	key = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpKeyDecrypt_internal( key, passphrase, pplen, hashedPhrase,
								   cacheTimeOut, cacheGlobal, inbuf,
								   inbuflen, format, outbuf, outbuflen );
}


	PGPInt32
pgpKeyVerify_back( PGPContextRef context, PGPUInt32 id, PGPByte const *inbuf,
	PGPSize inbuflen, PGPHashAlgorithm hashalg, PGPByte const *hash,
	PGPSize hashlen, PGPPublicKeyMessageFormat format )
{
	PGPKeyDBObj *	key;

	if (pgpRPCEnabled())
		return pgpKeyVerify_backRPC(context, id, inbuf, inbuflen, hashalg, hash, hashlen, format);
	key = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpKeyVerify_internal( key, inbuf, inbuflen, hashalg, hash,
								  hashlen, format );
}

	PGPError
pgpKeySign_back( PGPContextRef context, PGPUInt32 id,
	PGPByte const *passphrase, PGPSize pplen, PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	PGPHashAlgorithm hashalg, PGPByte const *hash, PGPSize hashlen,
	PGPPublicKeyMessageFormat format, PGPByte **outbuf, PGPSize *outbuflen )
{
	PGPKeyDBObj *	key;

	if (pgpRPCEnabled())
		return pgpKeySign_backRPC(context, id, passphrase, pplen, hashedPhrase,
					cacheTimeOut, cacheGlobal, hashalg, hash, hashlen, format,
					outbuf, outbuflen);
	key = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpKeySign_internal( key, passphrase, pplen, hashedPhrase,
								cacheTimeOut, cacheGlobal, hashalg,
								hash, hashlen, format, outbuf, outbuflen );
}


	PGPBoolean
pgpSecPassphraseOK_back( PGPContextRef context, PGPUInt32 id,
	PGPByte const *passphrase, PGPSize pplen, PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal)
{
	PGPKeyDBObj *	key;

	if (pgpRPCEnabled())
		return pgpSecPassphraseOK_backRPC( context, id, passphrase, pplen,
						hashedPhrase, cacheTimeOut, cacheGlobal );
	key = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpSecPassphraseOK_internal( key, passphrase, pplen, hashedPhrase,
										cacheTimeOut, cacheGlobal );
}

	PGPError
pgpKeyMaxSizes_back( PGPContextRef context, PGPUInt32 id,
	PGPUInt32 *maxEncryption, PGPUInt32 *maxDecryption,
	PGPUInt32 *maxSignature, PGPPublicKeyMessageFormat format )
{
	PGPKeyDBObj *	key;

	if (pgpRPCEnabled())
		return pgpKeyMaxSizes_backRPC( context, id, maxEncryption,
					maxDecryption, maxSignature, format);
	key = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpKeyMaxSizes_internal( key, maxEncryption, maxDecryption,
									maxSignature, format );
}

	PGPError
pgpSecProperties_back( PGPContextRef context, PGPUInt32 id,
	PGPBoolean *needsPassphrase, PGPBoolean *isSecretShared,
	PGPCipherAlgorithm *lockAlg, PGPUInt32 *lockBits )
{
	PGPKeyDBObj *	key;

	if (pgpRPCEnabled())
		return pgpSecProperties_backRPC( context, id, needsPassphrase, isSecretShared, lockAlg, lockBits );
	key = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpSecProperties_internal( key, needsPassphrase, isSecretShared,
									  lockAlg, lockBits );
}



	PGPError
pgpFetchKeyInfo_back( PGPContextRef context, PGPUInt32 id,
	PGPByte **bufptr, PGPSize *bufsize )
{
	PGPKeyDBObj *			obj;
	PGPKeyDBObjInfo *		info;
	PGPSize					infosize = 0;
	PGPByte *				buf;
	PGPUInt32				objtype;

	if (pgpRPCEnabled())
		return pgpFetchKeyInfo_backRPC( context, id, bufptr, bufsize );

	obj = (PGPKeyDBObj *)id;
	pgpAssert( pgpKeyDBObjIsValid( obj ) );
	info = obj->idinfo.info;
	objtype = pgpObjectType( obj );
	switch ( objtype )
	{
	case RINGTYPE_KEY:
	case RINGTYPE_SUBKEY:
		infosize = sizeof (PGPKeyInfo);
		break;
	case RINGTYPE_USERID:
		infosize = sizeof (PGPUserIDInfo);
		break;
	case RINGTYPE_SIG:
		infosize = sizeof (PGPSigInfo);
		break;
	case RINGTYPE_CRL:
		infosize = sizeof (PGPCRLInfo);
		break;
	default:
		infosize = sizeof (PGPUnkInfo);
		break;
	}
	buf = pgpContextMemAlloc( context, infosize, 0 );
	if( IsNull( buf ) )
		return kPGPError_OutOfMemory;
	pgpCopyMemory( (PGPByte *)info, buf, infosize );
	*bufptr = buf;
	*bufsize = infosize;
	return kPGPError_NoErr;
}


/*
 * Create a key array holding the keys and their children.
 * If set is specified, restrict to members of that set.  Use the
 * isMember function directly rather than calling pgpKeySetIsMember because
 * that eliminates deleted objects, and we need to return them here so that
 * the front end is guaranteed to be synched with us.
 */
	static PGPError
pgpCreateKeyArray( PGPContextRef context, PGPKeyDBObj *key, PGPUInt32 nkeys,
	PGPKeySetRef set, PGPUInt32 **objs, PGPSize *objsize )
{
	PGPKeyDBObj *		startkey;
	PGPKeyDBObj *		child;
	PGPKeyDBObj *		gchild;
	PGPUInt32			nentries;
	PGPUInt32			nchildren;
	PGPUInt32			i;
	PGPUInt32 *			buf;
	PGPUInt32 *			bp;

	startkey = key;
	nentries = 0;
	for( i=0; i<nkeys; ++i, key=key->next) 
	{
		pgpAssert( IsntNull( key ) );
		if( IsntNull( set ) && !set->isMember( set, key ) )
			continue;
		nentries += 3;
		for (child = key->down; IsntNull(child); child = child->next)
		{
			if( IsntNull( set ) && !set->isMember( set, child ) )
				continue;
			nentries += OBJISSIG(child) ? 2 : 3;
			for (gchild = child->down; IsntNull(gchild); gchild = gchild->next)
			{
				pgpAssert( OBJISSIG(gchild) );
				if( IsntNull( set ) && !set->isMember( set, gchild ) )
					continue;
				nentries += 2;
			}
		}
	}

	/* Allocate */
	
	buf = (PGPUInt32 *)pgpContextMemAlloc( context,
										   nentries * sizeof(PGPUInt32), 0 );
	if( IsNull( buf ) )
	{
		return kPGPError_OutOfMemory;
	}
	bp = buf;
	key = startkey;

	/* Flatten */
	
	for( i=0; i<nkeys; ++i, key=key->next)
	{
		if( IsntNull( set ) && !set->isMember( set, key ) )
			continue;
		*bp++ = (PGPUInt32) key;
		*bp++ = (PGPUInt32)key->objflags;
		nchildren = 0;
		for (child = key->down; IsntNull(child); child = child->next)
		{
			if( IsntNull( set ) && !set->isMember( set, child ) )
				continue;
			++nchildren;
		}
		*bp++ = nchildren;
		for (child = key->down; IsntNull(child); child = child->next)
		{
			if( IsntNull( set ) && !set->isMember( set, child ) )
				continue;
			*bp++ = (PGPUInt32) child;
			*bp++ = (PGPUInt32)child->objflags;
			if( OBJISSIG( child ) )
				continue;
			nchildren = 0;
			for (gchild = child->down; IsntNull(gchild); gchild = gchild->next)
			{
				if( IsntNull( set ) && !set->isMember( set, gchild ) )
					continue;
				++nchildren;
			}
			*bp++ = nchildren;
			for (gchild = child->down; IsntNull(gchild); gchild = gchild->next)
			{
				if( IsntNull( set ) && !set->isMember( set, gchild ) )
					continue;
				*bp++ = (PGPUInt32) gchild;
				*bp++ = (PGPUInt32)gchild->objflags;
			}
		}
	}

	*objs = buf;
	*objsize = nentries * sizeof(PGPUInt32);
	return kPGPError_NoErr;
}


	PGPError
pgpOpenKeyDBFile_back(
	PGPContextRef			context,
	PGPOpenKeyDBFileOptions	openFlags,
	PFLFileSpecRef			pubFileRef,
	PFLFileSpecRef			privFileRef,
	PGPUInt32 *				kdbid,
	PGPUInt32 *				numKeys,
	PGPUInt32 **			keyArray,
	PGPSize *				keyArraySize )
{
	PGPKeyDB *				kdb;
	PGPKeyDBObj *			key;
	PGPUInt32				nkeys;
	PGPError				err;

	if (pgpRPCEnabled())
		return pgpOpenKeyDBFile_backRPC( context, openFlags, pubFileRef, privFileRef,
					kdbid, numKeys, keyArray, keyArraySize );

	err = pgpOpenKeyDBFile_internal( context, openFlags, pubFileRef,
									 privFileRef, &kdb );

	if( IsPGPError( err ) )
		return err;

	/* Create keyarray for all the keys in the db (dummies too) */
	nkeys = 0;
	for (key = kdb->firstKeyInDB; IsntNull(key); key = key->next)
	{
		++nkeys;
	}
	err = pgpCreateKeyArray( context, kdb->firstKeyInDB, nkeys, NULL, keyArray,
							 keyArraySize );

	if( IsPGPError( err ) )
	{
		PGPFreeKeyDB( kdb );
		return err;
	}

	*kdbid = (PGPUInt32)kdb;
	*numKeys = nkeys;

	return kPGPError_NoErr;
}


	PGPError
pgpImportKeyBinary_back(
	PGPContextRef			context,
	PGPByte *				buffer,
	PGPSize					length,
	PGPUInt32 *				kdbid,
	PGPUInt32 *				numKeys,
	PGPUInt32 **			keyArray,
	PGPSize *				keyArraySize )
{
	PGPKeyDB *				kdb;
	PGPKeyDBObj *			key;
	PGPUInt32				nkeys;
	PGPError				err;

	if (pgpRPCEnabled())
		return pgpImportKeyBinary_backRPC( context, buffer, length, kdbid,
							numKeys, keyArray, keyArraySize );

	err = pgpImportKeyBinary_internal( context, buffer, length, &kdb );

	if( IsPGPError( err ) )
		return err;

	/* Create keyarray for all the keys in the db (dummies too) */
	nkeys = 0;
	for (key = kdb->firstKeyInDB; IsntNull(key); key = key->next)
	{
		++nkeys;
	}
	err = pgpCreateKeyArray( context, kdb->firstKeyInDB, nkeys, NULL,
							 keyArray, keyArraySize );

	if( IsPGPError( err ) )
	{
		PGPFreeKeyDB( kdb );
		return err;
	}

	*kdbid = (PGPUInt32)kdb;
	*numKeys = nkeys;

	return kPGPError_NoErr;
}

	PGPError
pgpKeyDBArray_back(
	PGPContextRef			context,
	PGPUInt32				kdbid,
	PGPUInt32 *				numKeys,
	PGPUInt32 **			keyArray,
	PGPSize *				keyArraySize )
{
	PGPKeyDB *				kdb = (PGPKeyDB *)kdbid;
	PGPKeyDBObj *			key;
	PGPUInt32				nkeys;
	PGPError				err;

	if (pgpRPCEnabled())
		return pgpKeyDBArray_backRPC( context, kdbid, numKeys,
							keyArray, keyArraySize );

	pgpAssert( pgpKeyDBIsValid( kdb ) );

	/* Create keyarray for all the keys in the db (dummies too) */
	nkeys = 0;
	for (key = kdb->firstKeyInDB; IsntNull(key); key = key->next)
	{
		++nkeys;
	}
	err = pgpCreateKeyArray( context, kdb->firstKeyInDB, nkeys, NULL,
							 keyArray, keyArraySize );

	if( IsPGPError( err ) )
	{
		PGPFreeKeyDB( kdb );
		return err;
	}

	*numKeys = nkeys;

	/*
	 * This function is only used when we have potentially made major
	 * changes to the keydb, hence add the whole thing to the notify list
	 */
	(void)pgpSetPendingNotify( NULL, NULL, pgpRootSet(kdb), NULL );

	return kPGPError_NoErr;
}

	PGPError
pgpUpdateKeyDB_back(
	PGPContextRef			context,
	PGPUInt32				kdbid,
	PGPUInt32 *				numNewKeys,
	PGPUInt32 **			newKeyArray,
	PGPSize *				newKeyArraySize,
	PGPUInt32 **			changedkeylist,
	PGPSize *				changedkeylistsize )
{
	PGPKeyDB *				kdb = (PGPKeyDB *)kdbid;
	PGPKeyDBObjRef			key;
	PGPUInt32				nkeys;
	PGPUInt32				nNewKeys;
	PGPKeySetRef			changedSet;
	PGPKeySetRef			newSet;
	PGPError				err;

	if (pgpRPCEnabled())
		return pgpUpdateKeyDB_backRPC( context, kdbid, numNewKeys,
							newKeyArray, newKeyArraySize,
							changedkeylist, changedkeylistsize );

	err = PGPNewEmptyKeySet( kdb, &newSet );
	if( IsPGPError( err ) )
		return err;
	err = PGPNewEmptyKeySet( kdb, &changedSet );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( newSet );
		return err;
	}
	err = pgpUpdateKeyDB_internal( kdb, newSet, changedSet );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( newSet );
		PGPFreeKeySet( changedSet );
		return err;
	}

	/* Create keyarray for the newSet */
	nkeys = nNewKeys = 0;
	for (key = kdb->firstKeyInDB; IsntNull(key); key = key->next)
	{
		++nkeys;
		/* See comment in pgpCreateKeyArray for why we use ->isMember */
		if( newSet->isMember( newSet, key ) )
			++nNewKeys;
	}
	err = pgpCreateKeyArray( context, kdb->firstKeyInDB, nkeys, newSet,
							 newKeyArray, newKeyArraySize );
	PGPFreeKeySet( newSet );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( changedSet );
		return err;
	}

	err = pgpKeySetFlattenFree(changedSet,changedkeylist,changedkeylistsize);
	if( IsPGPError( err ) )
	{
		PGPFreeData( newKeyArray );
		PGPFreeKeySet( changedSet );
		return err;
	}
	
	*numNewKeys = nNewKeys;
	return err;
}


	PGPError
pgpNewKeyDB_back( PGPContextRef context, PGPUInt32 *kdbid )
{
	PGPKeyDB *				kdb;
	PGPError				err;

	if (pgpRPCEnabled())
		return pgpNewKeyDB_backRPC( context, kdbid );

	err = pgpNewKeyDB_internal( context, &kdb );

	if( IsPGPError( err ) )
		return err;

	*kdbid = (PGPUInt32)kdb;

	return kPGPError_NoErr;
}


	PGPError
pgpKeyDBFlush_back( PGPContextRef context, PGPUInt32 id,
	PGPUInt32 **changedkeylist, PGPSize *changedkeylistsize )
{
	PGPKeyDBRef kdb;
	PGPKeySetRef changedSet;
	PGPError err;

	if (pgpRPCEnabled())
		return pgpKeyDBFlush_backRPC( context, id, changedkeylist, changedkeylistsize );

	kdb = (PGPKeyDB *)id;
	pgpAssert( pgpKeyDBIsValid( kdb ) );
	err = PGPNewEmptyKeySet( kdb, &changedSet );
	if( IsPGPError( err ) )
		return err;

	err = pgpKeyDBFlush_internal( kdb, changedSet );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( changedSet );
		return err;
	}

	(void)pgpSetPendingNotify( changedSet, NULL, NULL, NULL );
	return pgpKeySetFlattenFree(changedSet,changedkeylist,changedkeylistsize);
}

	void
pgpFreeKeyDB_back( PGPContextRef context, PGPUInt32 id )
{
	if (pgpRPCEnabled()) {
		pgpFreeKeyDB_backRPC( context, id );
		return;
	}
	pgpKeyDBDestroy_internal( (PGPKeyDB *)id );
}


	PGPError
pgpSetKeyEnabled_back( PGPContextRef context, PGPUInt32 id, PGPBoolean enable )
{
	PGPKeyDBObj *key = (PGPKeyDBObj *)id;

	if (pgpRPCEnabled())
		return pgpSetKeyEnabled_backRPC( context, id , enable );
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	if( enable )
		return pgpEnableKey( key );
	else
		return pgpDisableKey( key );
}

	PGPError
pgpSetKeyAxiomatic_back (
	PGPContextRef	context,
	PGPUInt32		id,
	PGPBoolean		setAxiomatic,
	PGPBoolean		checkPassphrase,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal
	)
{
	PGPKeyDBObj *key = (PGPKeyDBObj *)id;
	if (pgpRPCEnabled())
		return pgpSetKeyAxiomatic_backRPC( context, id , setAxiomatic,
				checkPassphrase, passphrase, passphraseLength, hashedPhrase,
				cacheTimeOut, cacheGlobal );
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpSetKeyAxiomatic_internal( key, setAxiomatic, checkPassphrase,
										passphrase, passphraseLength,
										hashedPhrase, cacheTimeOut,
										cacheGlobal );
}


	PGPError
pgpPropagateTrust_back( PGPContextRef context, PGPUInt32 setid,
	PGPUInt32 *keylist, PGPSize keylistsize, PGPUInt32 altid,
	PGPUInt32 const timenow, PGPUInt32 **changedkeylist,
	PGPSize *changedkeylistsize )
{
	PGPKeyDBRef		setdb, altdb;
	PGPKeySetRef	set;
	PGPKeySetRef	changedSet;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpPropagateTrust_backRPC( context, setid, keylist, keylistsize, altid, timenow, changedkeylist, changedkeylistsize );
	setdb = (PGPKeyDBRef) setid;
	altdb = (PGPKeyDBRef) altid;
	pgpAssert( pgpKeyDBIsValid( setdb ) );
	pgpAssert( IsNull( altdb ) || pgpKeyDBIsValid( altdb ) );

	err = pgpKeySetUnflattenFree( setdb, keylist, keylistsize, &set );
	if( IsPGPError( err ) )
		return err;

	err = PGPNewEmptyKeySet( setdb, &changedSet );
	if( IsPGPError( err ) )
	{
		PGPFreeData( keylist );
		return err;
	}
	err = pgpPropagateTrustInternal( set, altdb, timenow, changedSet );
	PGPFreeKeySet( set );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( changedSet );
		return err;
	}

	(void)pgpSetPendingNotify( changedSet, NULL, NULL, NULL );
	return pgpKeySetFlattenFree(changedSet,changedkeylist,changedkeylistsize);
}


	PGPError
pgpCheckKeyRingSigs_back( PGPContextRef context, PGPUInt32 setid,
	PGPUInt32 *keylist, PGPSize keylistsize, PGPUInt32 altid,
	PGPBoolean checkAll, PGPUInt32 **changedkeylist,
	PGPSize *changedkeylistsize )
{
	PGPKeyDBRef		setdb, altdb;
	PGPKeySetRef	set;
	PGPKeySetRef	changedSet;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpCheckKeyRingSigs_backRPC( context, setid, keylist, keylistsize, altid, checkAll, changedkeylist, changedkeylistsize );
	setdb = (PGPKeyDBRef) setid;
	altdb = (PGPKeyDBRef) altid;
	pgpAssert( pgpKeyDBIsValid( setdb ) );
	pgpAssert( IsNull( altdb ) || pgpKeyDBIsValid( altdb ) );

	err = pgpKeySetUnflattenFree( setdb, keylist, keylistsize, &set );
	if( IsPGPError( err ) )
		return err;

	err = PGPNewEmptyKeySet( setdb, &changedSet );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( set );
		return err;
	}
	err = pgpCheckKeyRingSigs_internal( set, altdb, checkAll, NULL, 0,
										changedSet );
	PGPFreeKeySet( set );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( changedSet );
		return err;
	}

	(void)pgpSetPendingNotify( changedSet, NULL, NULL, NULL );
	return pgpKeySetFlattenFree(changedSet,changedkeylist,changedkeylistsize);
}


	PGPError
pgpPrepareToCheckKeyRingSigs_back( PGPContextRef context, PGPUInt32 setid,
	PGPUInt32 *keylist, PGPSize keylistsize, PGPUInt32 altid,
	PGPBoolean checkAll, PGPUInt32 *nsigs, PGPUInt32 **changedkeylist,
	PGPSize *changedkeylistsize )
{
	PGPKeyDBRef		setdb, altdb;
	PGPKeySetRef	set;
	PGPKeySetRef	changedSet;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpPrepareToCheckKeyRingSigs_backRPC( context, setid, keylist, keylistsize,
					altid, checkAll, nsigs, changedkeylist, changedkeylistsize );
	setdb = (PGPKeyDBRef) setid;
	altdb = (PGPKeyDBRef) altid;
	pgpAssert( pgpKeyDBIsValid( setdb ) );
	pgpAssert( IsNull( altdb ) || pgpKeyDBIsValid( altdb ) );

	err = pgpKeySetUnflattenFree( setdb, keylist, keylistsize, &set );
	if( IsPGPError( err ) )
		return err;

	err = PGPNewEmptyKeySet( setdb, &changedSet );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( set );
		return err;
	}
	err = pgpPrepareToCheckKeyRingSigs_internal( set, altdb, checkAll, nsigs,
												 changedSet );
	PGPFreeKeySet( set );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( changedSet );
		return err;
	}

	(void)pgpSetPendingNotify( changedSet, NULL, NULL, NULL );
	return pgpKeySetFlattenFree(changedSet,changedkeylist,changedkeylistsize);
}


	PGPError
pgpCheckSig_back( PGPContextRef context, PGPUInt32 setid, PGPUInt32 objid,
	PGPUInt32 altid, PGPBoolean checkAll, PGPBoolean revocationonly,
	PGPBoolean *handled, PGPBoolean *changed, PGPBoolean *verified )
{
	PGPKeyDBRef		setdb, altdb;
	PGPKeyDBObjRef	obj;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpCheckSig_backRPC( context, setid, objid, altid, checkAll, revocationonly, handled, changed, verified );
	setdb = (PGPKeyDBRef) setid;
	altdb = (PGPKeyDBRef) altid;
	obj = (PGPKeyDBObj *)objid;
	pgpAssert( pgpKeyDBObjIsValid( obj ) );
	pgpAssert( pgpKeyDBIsValid( setdb ) );
	pgpAssert( IsNull( altdb ) || pgpKeyDBIsValid( altdb ) );

	err = pgpCheckSig_internal( setdb, obj, altdb, checkAll, revocationonly,
								handled, changed, verified );
	return err;
}



	PGPError
pgpDoGenerateKey_back (
	PGPContextRef	context,
	PGPUInt32		keydbID,
	PGPUInt32		masterkeyID,
	PGPByte			pkalg,
	PGPUInt32		bits,
	PGPTime			creationDate,
	PGPUInt16		expirationDays,
	char const *	name,
	PGPSize			name_len, 
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		passphraseIsKey,
	char const *	masterpass, 
	PGPSize			masterpassLength,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPEventHandlerProcPtr progress,
	PGPUserValue	userValue,
	PGPBoolean		fastgen,
	PGPBoolean		checkentropy,
	PGPBoolean		usetoken,
	PGPUInt32		tokenid,
	PGPUInt32 *		adkkeylist,
	PGPSize			adkkeylistsize,
	PGPByte			adkclass,
	PGPUInt32 *		rakkeylist,
	PGPSize			rakkeylistsize,
	PGPByte			rakclass,
	PGPCipherAlgorithm const * prefalg,
	PGPSize			prefalgLength,
	PGPByte const *	prefkeyserv,
	PGPSize			prefkeyservLength,
	PGPUInt32		keyflags,
	PGPBoolean		fkeyflags,
	PGPUInt32		keyservprefs,
	PGPBoolean		fkeyservprefs,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength,
	PGPUInt32 *		newKey)
{
	PGPKeyDB *		db;
	PGPKeySetRef	adkset;
	PGPKeySetRef	rakset;
	PGPKeyDBObjRef	key;
	PGPKeyDBObjRef	masterkey;
	PGPError		err;

	if (pgpRPCEnabled())
	{
		return pgpDoGenerateKey_backRPC( context, keydbID, masterkeyID, pkalg,
				bits, creationDate, expirationDays, name, name_len, passphrase,
				passphraseLength, passphraseIsKey, masterpass,
				masterpassLength, cacheTimeOut, cacheGlobal, progress,
				userValue, fastgen, checkentropy, usetoken, tokenid,
				adkkeylist, adkkeylistsize, adkclass,
				rakkeylist, rakkeylistsize, rakclass, prefalg, prefalgLength,
				prefkeyserv, prefkeyservLength, keyflags, fkeyflags,
				keyservprefs, fkeyservprefs, newobjs, newobjslength, newKey );
	}
	db = (PGPKeyDB *)keydbID;
	pgpAssert( pgpKeyDBIsValid( db ) );
	masterkey = (PGPKeyDBObj *) masterkeyID;
	pgpAssert( IsNull(masterkey) || pgpKeyDBObjIsValid( masterkey ) );
	
	err = pgpKeySetUnflattenFree( db, adkkeylist, adkkeylistsize, &adkset );
	if( IsPGPError( err ) )
		return err;

	err = pgpKeySetUnflattenFree( db, rakkeylist, rakkeylistsize, &rakset );
	if( IsPGPError( err ) )
	{
		PGPFreeKeySet( adkset );
		return err;
	}

#if PGP_WIN32
	/* Can't do callbacks in win32, use our special function above */
	progress = NULL;
#endif	
	err = pgpDoGenerateKey_internal( db, masterkey, pkalg, bits, creationDate,
					expirationDays, name, name_len, passphrase,
					passphraseLength, passphraseIsKey, masterpass,
					masterpassLength, cacheTimeOut, cacheGlobal, progress,
					userValue, fastgen, checkentropy, usetoken, tokenid,
					adkset, adkclass, rakset, rakclass, prefalg, prefalgLength,
					prefkeyserv, prefkeyservLength, keyflags, fkeyflags,
					keyservprefs, fkeyservprefs, &key );
	PGPFreeKeySet( adkset );
	PGPFreeKeySet( rakset );
	if( IsPGPError( err ) )
		return err;
	*newKey = (PGPUInt32)key;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, key );
	return pgpCreateKeyArray( context, key, 1, NULL, newobjs, newobjslength );
}


	PGPError
pgpCreateKeypair_back (
	PGPContextRef	context,
	PGPUInt32		keydbID,
	PGPByte *		seckeyBuf,
	PGPSize			seckeySize,
	PGPByte *		keyspecBuf,
	PGPSize			keyspecSize,
	char const *	name,
	PGPSize			name_len, 
	PGPUInt32 *		adklist,
	PGPSize			adklistsize,
	PGPByte			adkclass,
	PGPUInt32 *		raklist,
	PGPSize			raklistsize,
	PGPByte			rakclass,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		passphraseIsKey,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength,
	PGPUInt32 *		newkeyid )
{
	PGPKeyDB *		db;
	PGPKeySetRef	adkset;
	PGPKeySetRef	rakset;
	PGPSecKey *		seckey;
	PGPKeySpec *	keyspec;
	PGPEnv *		pgpEnv;
	PGPKeyDBObjRef	key;
	PGPBoolean		v3;
	PGPByte			pkalg;
	PGPError		err;

	if (pgpRPCEnabled())
	{
		return pgpCreateKeypair_backRPC( context, keydbID,
							seckeyBuf, seckeySize, keyspecBuf, keyspecSize,
							name, name_len,
							adklist, adklistsize, adkclass,
							raklist, raklistsize, rakclass,
							passphrase, passphraseLength,
							passphraseIsKey, cacheTimeOut, cacheGlobal,
							newobjs, newobjslength, newkeyid);
	}

	db = (PGPKeyDB *)keydbID;
	pgpAssert( pgpKeyDBIsValid( db ) );

	keyspec = pgpKeySpecFromBuf( context, keyspecBuf, keyspecSize, &err );
	PGPFreeData( keyspecBuf );
	if( IsPGPError( err ) )
	{
		PGPFreeData( seckeyBuf );
		PGPFreeData( adklist );
		PGPFreeData( raklist );
		return err;
	}
	
	v3 = pgpKeySpecVersion( keyspec ) < PGPVERSION_4;
	pkalg = pgpKeySpecPkAlg( keyspec );
	seckey = pgpSecKeyFromBuf( context, pkalg, seckeyBuf, seckeySize,
							   v3, NULL, &err );
	PGPFreeData( seckeyBuf );
	if( IsPGPError( err ) )
	{
		pgpKeySpecDestroy( keyspec );
		PGPFreeData( adklist );
		PGPFreeData( raklist );
		return err;
	}

	err = pgpKeySetUnflattenFree( db, adklist, adklistsize, &adkset );
	if( IsPGPError( err ) )
	{
		pgpSecKeyDestroy( seckey );
		pgpKeySpecDestroy( keyspec );
		PGPFreeKeySet( adkset );
		PGPFreeData( raklist );
		return err;
	}

	err = pgpKeySetUnflattenFree( db, raklist, raklistsize, &rakset );
	if( IsPGPError( err ) )
	{
		pgpSecKeyDestroy( seckey );
		pgpKeySpecDestroy( keyspec );
		PGPFreeKeySet( adkset );
		return err;
	}

	/* Add to cache if requested */
	if( IsntNull( passphrase ) )
		(void)pgpSecKeyCachePassphrase( seckey, (PGPByte const *)passphrase,
										passphraseLength, passphraseIsKey,
										cacheTimeOut, cacheGlobal );


	pgpEnv = pgpContextGetEnvironment( context );
	key = pgpCreateKeypair (pgpEnv, db, seckey, keyspec,
					name, name_len,
					adkset, adkclass, rakset, rakclass,
					(PGPByte *) passphrase, passphraseLength,
					passphraseIsKey, &err);

	pgpSecKeyDestroy( seckey );
	pgpKeySpecDestroy( keyspec );
	PGPFreeKeySet( adkset );
	PGPFreeKeySet( rakset );
	if( IsPGPError( err ) )
		return err;
	*newkeyid = (PGPUInt32)key;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, key );
	return pgpCreateKeyArray( context, key, 1, NULL, newobjs, newobjslength );
}

	PGPError
pgpCreateSubkeypair_back (
	PGPContextRef	context,
	PGPUInt32		masterkeyID,
	PGPByte *		seckeyBuf,
	PGPSize			seckeySize,
	PGPByte *		keyspecBuf,
	PGPSize			keyspecSize,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		passphraseIsKey,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength,
	PGPUInt32 *		newkeyid )
{
	PGPKeyDB *		db;
	PGPKeyDBObjRef	masterkey;
	PGPSecKey *		seckey;
	PGPKeySpec *	keyspec;
	PGPKeyDBObjRef	subkey;
	PGPEnv *		pgpEnv;
	PGPBoolean		v3;
	PGPByte			pkalg;
	PGPError		err;

	if (pgpRPCEnabled())
	{
		return pgpCreateSubkeypair_backRPC( context, masterkeyID,
								seckeyBuf, seckeySize,
								keyspecBuf, keyspecSize,
								passphrase, passphraseLength,
								passphraseIsKey, cacheTimeOut, cacheGlobal,
								newobjs, newobjslength, newkeyid );
	}

	masterkey = (PGPKeyDBObj *) masterkeyID;
	pgpAssert( IsNull(masterkey) || pgpKeyDBObjIsValid( masterkey ) );
	db = PGPPeekKeyDBObjKeyDB( masterkey );
	pgpAssert( pgpKeyDBIsValid( db ) );

	keyspec = pgpKeySpecFromBuf( context, keyspecBuf, keyspecSize, &err );
	PGPFreeData( keyspecBuf );
	if( IsPGPError( err ) )
	{
		PGPFreeData( seckeyBuf );
		return err;
	}
	
	v3 = pgpKeySpecVersion( keyspec ) < PGPVERSION_4;
	pkalg = pgpKeySpecPkAlg( keyspec );
	seckey = pgpSecKeyFromBuf( context, pkalg, seckeyBuf, seckeySize,
							   v3, NULL, &err );
	PGPFreeData( seckeyBuf );
	if( IsPGPError( err ) )
	{
		pgpKeySpecDestroy( keyspec );
		return err;
	}

	/* Add to cache if requested */
	if( IsntNull( passphrase ) )
		(void)pgpSecKeyCachePassphrase( seckey, (PGPByte const *)passphrase,
										passphraseLength, passphraseIsKey,
										cacheTimeOut, cacheGlobal );


	pgpEnv = pgpContextGetEnvironment( context );
	subkey = pgpCreateSubkeypair (masterkey, pgpEnv, seckey,
							 keyspec, (PGPByte const *)passphrase,
							 passphraseLength, passphraseIsKey, &err );

	pgpSecKeyDestroy( seckey );
	pgpKeySpecDestroy( keyspec );
	if( IsPGPError( err ) )
		return err;
	*newkeyid = (PGPUInt32)subkey;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, masterkey );
	return pgpCreateKeyArray( context, subkey, 1, NULL,
							  newobjs, newobjslength );
}

	PGPError
pgpCountTokens_back (
	PGPContextRef	context,
	PGPUInt32		*numTokens
	)
{
	if (pgpRPCEnabled())
		return pgpCountTokens_backRPC( context, numTokens );
	return pgpCountTokens_internal( context, numTokens );
}

	PGPError
pgpGetTokenInfo_back (
	PGPContextRef	context,
    PGPUInt32       tokNumber, 
	PGPTokenInfo		*tokenInfo )
{
	if (pgpRPCEnabled())
		return pgpGetTokenInfo_backRPC( context, tokNumber, tokenInfo );
	return pgpGetTokenInfo_internal( context, tokNumber, tokenInfo );
}

	PGPError
pgpDeleteKeyOnToken_back (
	PGPContextRef	context,
	PGPUInt32 keydbID, 
	const PGPKeyID *keyID, 
	PGPUInt32 tokNumber, 
	PGPByte const *pin, PGPSize pinLen )
{
	if (pgpRPCEnabled())
		return pgpDeleteKeyOnToken_backRPC( context, keydbID, keyID, tokNumber, pin, pinLen );

	return pgpDeleteKeyOnToken_internal( context, keydbID, keyID, tokNumber, pin, pinLen );
}


	PGPError
pgpWipeToken_back (
	PGPContextRef	context,
	PGPUInt32		tokNumber,
	PGPByte const *	passphrase,
	PGPSize			passphraseLength
	)
{
	if (pgpRPCEnabled())
		return pgpWipeToken_backRPC( context, tokNumber, passphrase,
									 passphraseLength );
	return pgpWipeToken_internal( context, tokNumber, passphrase,
								  passphraseLength );
}

	PGPError
pgpTokenPassphraseIsValid_back (
	PGPContextRef	context,
	PGPUInt32		tokNumber,
	PGPByte const *	passphrase,
	PGPSize			passphraseLength
	)
{
	if (pgpRPCEnabled())
		return pgpTokenPassphraseIsValid_backRPC( context, tokNumber,
											passphrase, passphraseLength );
	return pgpTokenPassphraseIsValid_internal( context, tokNumber, passphrase,
											   passphraseLength );
}

	PGPError
pgpCopyKeyToToken_back (
	PGPContextRef	context,
	PGPUInt32		objid,
	PGPUInt32		tokNumber,
	PGPBoolean		isMaster,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	char const *	PIN,
	PGPSize			PINlength,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal)
{
	PGPKeyDBObj *	key = (PGPKeyDBObj *) objid;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpCopyKeyToToken_backRPC( context, objid, tokNumber,
				isMaster,
				passphrase, passphraseLength, hashedPhrase, PIN, PINlength,
				cacheTimeOut, cacheGlobal);
	pgpAssert( pgpKeyIsValid( key ) );
	err = pgpCopyKeyToToken_internal( key, tokNumber, isMaster, passphrase,
								passphraseLength, hashedPhrase, PIN, PINlength,
								cacheTimeOut, cacheGlobal );

	return err;
}
   	
   	PGPError
pgpTokenImportX509_back (
	PGPContextRef	context,
	const PGPByte         *keyID,
    const PGPByte         *userID,    PGPSize userIDlen, 
    const PGPByte         *x509,      PGPSize x509len, 
    const PGPByte         *password,  PGPSize passwordLength )
{
	if( pgpRPCEnabled() )
		return pgpTokenImportX509_backRPC( 
            context, keyID, userID, userIDlen, 
            x509, x509len, password, passwordLength );
	return pgpTokenImportX509_internal( 
            context, keyID, userID, userIDlen, 
            x509, x509len, password, passwordLength );
}


   	PGPError
pgpTokenPutKeyContainer_back (
    PGPContextRef   context, 
	const PGPByte   *keyID,
    const PGPByte   *password,	PGPSize passwordLen, 
    const PGPByte   *cont,		PGPSize contSize )
{
    if( pgpRPCEnabled() ) 
        return pgpTokenPutKeyContainer_backRPC(
           context, keyID, password, passwordLen, cont, contSize );

	return pgpTokenPutKeyContainer_internal( 
           context, keyID, password, passwordLen, cont, contSize );
}


   	PGPError
pgpTokenGetKeyContainer_back (
    PGPContextRef   context, 
	const PGPByte   *keyID,
    const PGPByte   *password,	PGPSize passwordSize, 
    PGPByte			**contOut,	PGPSize *contOutSize )
{
    if( pgpRPCEnabled() ) 
        return pgpTokenGetKeyContainer_backRPC(
           context, keyID, password, passwordSize, 
		   contOut, contOutSize );

	return pgpTokenGetKeyContainer_internal( 
           context, keyID, password, passwordSize, 
		   contOut, contOutSize );
}

static PGPError pgpSetPKCS11DrvFile_internal( PGPByte *module )  
{
    if( module && *module )  {
        if( sP11Module[0] != '\0' )  {
#ifdef PGP_WIN32
            /* On Windows filenames are case insensitive */
            if( stricmp( sP11Module, module ) != 0 )  
#else
            if( strcmp( (char *) sP11Module, (char *) module ) != 0 )  
#endif
            {
                pgpUnloadTCL();
                strcpy((char *) sP11Module, (char *) module);
            }
        }
        else
            strcpy((char *) sP11Module, (char *) module);
    }
    else  {
        pgpUnloadTCL();
		sP11Module[0] = '\0';
	}
    return kPGPError_NoErr;
}


    PGPError 
pgpSetPKCS11DrvFile_back( PGPByte *module )  
{
	if (pgpRPCEnabled())
		return pgpSetPKCS11DrvFile_backRPC( module );
	return pgpSetPKCS11DrvFile_internal( module );
}

	PGPError
pgpAddUserID_back (
	PGPContextRef	context,
	PGPUInt32		objid,
	PGPBoolean		isAttribute,
	PGPAttributeType attributeType,
	char const *	userIDData,
	PGPSize		   	userIDLength,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength)
{
	PGPKeyDBObj *	key = (PGPKeyDBObj *) objid;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpAddUserID_backRPC( context, objid, isAttribute,
				attributeType, userIDData, userIDLength, passphrase,
				passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal,
				newobjs, newobjslength);
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	err = pgpAddUserID_internal( key, isAttribute, attributeType, userIDData,
								 userIDLength, passphrase, passphraseLength,
								 hashedPhrase, cacheTimeOut, cacheGlobal );

	if( IsPGPError( err ) )
		return err;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, key );
	return pgpCreateKeyArray( context, key, 1, NULL, newobjs, newobjslength );
}


	PGPError
pgpCertifyUserID_back(
	PGPContextRef	context,
	PGPUInt32		useridid,
	PGPUInt32		certifying_keyid,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPBoolean		exportable,
	PGPTime			creationDate,
	PGPUInt32		expiration,
	PGPByte			trustDepth,
	PGPByte			trustValue,
	char const *	sRegExp,
	PGPSize			sRegExpLength,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength
	)
{
	PGPKeyDBObj *	userid = (PGPKeyDBObj *) useridid;
	PGPKeyDBObj *	certifying_key = (PGPKeyDBObj *) certifying_keyid;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpCertifyUserID_backRPC( context, useridid, certifying_keyid,
				passphrase, passphraseLength, hashedPhrase, cacheTimeOut,
				cacheGlobal, exportable, creationDate, expiration, 
				trustDepth, trustValue, sRegExp, sRegExpLength, newobjs,
				newobjslength);

	pgpAssert( pgpKeyDBObjIsValid( userid ) );
	pgpAssert( pgpKeyDBObjIsValid( certifying_key ) );
	err = pgpCertifyUserID_internal( userid, certifying_key, passphrase,
									 passphraseLength, hashedPhrase,
									 cacheTimeOut, cacheGlobal, exportable,
									 creationDate, expiration,
									 trustDepth, trustValue, sRegExp );
	if( IsPGPError( err ) )
		return err;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, PGPPeekKeyDBObjKey(userid) );
	return pgpCreateKeyArray( context, userid, 1, NULL,
							  newobjs, newobjslength );
}

	PGPError
pgpCertifyPrimaryUserID_back (PGPContextRef context, PGPUInt32 useridid,
	char *passphrase, PGPSize passphraseLength, PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal, PGPUInt32 **newobjs,
	PGPSize *newobjslength)
{
	PGPKeyDBObj *	userid = (PGPKeyDBObj *) useridid;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpCertifyPrimaryUserID_backRPC( context, useridid, passphrase,
				passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal,
				newobjs, newobjslength );
	pgpAssert( pgpKeyDBObjIsValid( userid ) );
	err = pgpCertifyPrimaryUserID_internal( userid, passphrase,
											passphraseLength, hashedPhrase,
											cacheTimeOut, cacheGlobal );
	if( IsPGPError( err ) )
		return err;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, PGPPeekKeyDBObjKey(userid) );
	return pgpCreateKeyArray( context, userid->up, 1, NULL,
							  newobjs, newobjslength );
}


	PGPError
pgpRevokeSig_back (
	PGPContextRef	context,
	PGPUInt32		sigid,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength
	)
{
	PGPKeyDBObj *	sig = (PGPKeyDBObj *) sigid;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpRevokeSig_backRPC( context, sigid, passphrase,
				passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal,
				newobjs, newobjslength );
	pgpAssert( pgpKeyDBObjIsValid( sig ) );
	err = pgpRevokeSig_internal( context, sig, passphrase, passphraseLength,
								 hashedPhrase, cacheTimeOut, cacheGlobal );
	if( IsPGPError( err ) )
		return err;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, PGPPeekKeyDBObjKey(sig) );
	return pgpCreateKeyArray( context, sig->up, 1, NULL,
							  newobjs, newobjslength );
}

/* This is also called to revoke subkeys */
	PGPError
pgpRevokeKey_back (
	PGPContextRef		context,
	PGPUInt32			key_id,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPBoolean			hashedPhrase,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal,
	PGPUInt32 **		newobjs,
	PGPSize *			newobjslength
	)
{
	PGPKeyDBObj *	keyorsubkey = (PGPKeyDBObj *) key_id;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpRevokeKey_backRPC( context, key_id, passphrase,
				passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal, 
				newobjs, newobjslength);
	pgpAssert( pgpKeyDBObjIsValid( keyorsubkey ) );
	err = pgpRevokeKey_internal( keyorsubkey, passphrase, passphraseLength,
								 hashedPhrase, cacheTimeOut, cacheGlobal );
	if( IsPGPError( err ) )
		return err;
	(void)pgpSetPendingNotify( NULL, NULL, NULL,
							   PGPPeekKeyDBObjKey(keyorsubkey) );
	return pgpCreateKeyArray( context, keyorsubkey, 1, NULL,
							  newobjs, newobjslength );
}

	PGPError
pgpDoChangePassphrase_back (PGPContextRef context, PGPUInt32 keydb_id,
							PGPUInt32 key_id, PGPUInt32 masterkey_id, 
							const char *oldphrase, PGPSize oldphraseLength,
							const char *newphrase, PGPSize newphraseLength,
							PGPBoolean newPassphraseIsKey,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal)
{
	PGPKeyDB *		keydb = (PGPKeyDB *)keydb_id;
	PGPKeyDBObj *	key = (PGPKeyDBObj *) key_id;
	PGPKeyDBObj *	masterkey = (PGPKeyDBObj *) masterkey_id;

	if (pgpRPCEnabled())
		return pgpDoChangePassphrase_backRPC( context, keydb_id, key_id,
					masterkey_id, oldphrase, oldphraseLength, newphrase,
					newphraseLength, newPassphraseIsKey, cacheTimeOut,
					cacheGlobal );
	pgpAssert( pgpKeyDBIsValid( keydb ) );
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	pgpAssert( IsNull(masterkey) ||  pgpKeyDBObjIsValid( masterkey ) );

	return pgpDoChangePassphrase_internal( keydb, key, masterkey, oldphrase,
										   oldphraseLength, newphrase,
										   newphraseLength,
										   newPassphraseIsKey, cacheTimeOut,
										   cacheGlobal );
}

	PGPError
pgpPassphraseIsValid_back(
	PGPContextRef	context,
	PGPUInt32		key_id,
	const char *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPBoolean *	isValid)
{
	PGPKeyDBObj *	key = (PGPKeyDBObj *) key_id;

	if (pgpRPCEnabled())
		return pgpPassphraseIsValid_backRPC( context, key_id, passphrase,
				passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal,
				isValid );
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpPassphraseIsValid_internal( key, passphrase, passphraseLength,
										  hashedPhrase, cacheTimeOut,
										  cacheGlobal, isValid );
}


	PGPError
pgpPurgePassphraseCache_back(
	PGPContextRef	context)
{
	if (pgpRPCEnabled())
		return pgpPurgePassphraseCache_backRPC( context );
	return pgpPurgePassphraseCache_internal( context );
}


	PGPError
pgpSetKeyTrust_back(
	PGPContextRef	context,
	PGPUInt32		key_id,
	PGPUInt32		trust)
{
	PGPKeyDBObj *	key = (PGPKeyDBObj *) key_id;

	if (pgpRPCEnabled())
		return pgpSetKeyTrust_backRPC( context, key_id, trust );
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	return pgpSetKeyTrust_internal( key, trust );
}

	PGPError
pgpGetPasskeyBuffer_back (
	PGPContextRef		context,
	PGPUInt32			key_id,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPByte				**passkeyBuffer,
	PGPSize				*passkeyBufferLength
	)
{
	PGPKeyDBObj *	key = (PGPKeyDBObj *) key_id;
	PGPByte *		buf;
	PGPInt32		bufsize;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpGetPasskeyBuffer_backRPC( context, key_id, passphrase, passphraseLength,
					passkeyBuffer, passkeyBufferLength );
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	*passkeyBuffer = NULL;
	*passkeyBufferLength = 0;

	if( OBJISSUBKEY( key ))
		pgpGetSubKeyNumberInternal( key, kPGPSubKeyProperty_LockingBits,
									&bufsize );
	else
		pgpGetKeyNumberInternal( key, kPGPKeyProperty_LockingBits, &bufsize );
	bufsize /= 8;
	buf = pgpContextMemAlloc( context, bufsize, 0 );
	if( IsNull( buf ) )
		return kPGPError_OutOfMemory;
	err = pgpGetPasskeyBuffer_internal( key, passphrase, passphraseLength,buf);
	if( IsPGPError( err ) )
	{
		PGPFreeData( buf );
		return err;
	}

	*passkeyBuffer = buf;
	*passkeyBufferLength = bufsize;
	return kPGPError_NoErr;
}


	PGPError
pgpAddKeyOptions_back (PGPContextRef context, PGPUInt32 key_id,
	char * passphrase, PGPSize passphraseLength, PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	PGPUInt32 *raklist, PGPSize raklistsize, PGPUInt32 rakclass,
	PGPUInt32 **newobjs, PGPSize *newobjslength)
{
	PGPKeyDBObj *	key = (PGPKeyDBObj *) key_id;
	PGPKeyDB *		db;
	PGPKeySetRef	rakset;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpAddKeyOptions_backRPC( context, key_id, passphrase,
					passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal, 
					raklist, raklistsize, rakclass, newobjs, newobjslength );
	pgpAssert( pgpKeyDBObjIsValid( key ) );
	db = PGPPeekKeyDBObjKeyDB( key );
	err = pgpKeySetUnflattenFree( db, raklist, raklistsize, &rakset );
	if( IsPGPError( err ) )
		return err;

	err = pgpAddKeyOptions_internal( key, passphrase, passphraseLength,
									 hashedPhrase, cacheTimeOut, cacheGlobal,
									 rakset, rakclass );
	PGPFreeKeySet( rakset );
	if( IsPGPError( err ) )
		return err;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, key );
	return pgpCreateKeyArray( context, key, 1, NULL, newobjs, newobjslength );
}


	PGPError
pgpUpdateKeyOptions_back (
	PGPContextRef		context,
	PGPUInt32			key_id,
	char *				passphrase,
	PGPSize				passphraseLength,
	PGPBoolean			hashedPhrase,
	PGPUInt32			cacheTimeOut,
	 PGPBoolean			cacheGlobal,
	PGPCipherAlgorithm	*prefalg,
	PGPSize				prefalgLength,
	PGPByte				*prefkeyserv,
	PGPSize				prefkeyservLength,
	PGPUInt32			keyflags,
	PGPBoolean			fkeyflags,
	PGPUInt32			keyservprefs,
	PGPBoolean			fkeyservprefs,
	PGPUInt32 **		newobjs,
	PGPSize *			newobjslength
	)
{
	PGPKeyDBObj *	key = (PGPKeyDBObj *) key_id;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpUpdateKeyOptions_backRPC( context, key_id, passphrase,
					passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal,
					prefalg, prefalgLength, prefkeyserv, prefkeyservLength,
					keyflags, fkeyflags, keyservprefs, fkeyservprefs,
					newobjs, newobjslength );
	pgpAssert( pgpKeyDBObjIsValid( key ) );

	err = pgpUpdateKeyOptions_internal( key, passphrase, passphraseLength,
										hashedPhrase, cacheTimeOut,
										cacheGlobal, prefalg, prefalgLength,
										prefkeyserv, prefkeyservLength,
										keyflags, fkeyflags, keyservprefs,
										fkeyservprefs);
	if( IsPGPError( err ) )
		return err;
	(void)pgpSetPendingNotify( NULL, NULL, NULL, key );
	return pgpCreateKeyArray( context, key, 1, NULL, newobjs, newobjslength );
}

	PGPError
pgpKeyDBAddObject_back( PGPContextRef context, PGPUInt32 kdb_id,
	PGPUInt32 obj_id, PGPUInt32 **newobjs, PGPSize *newobjslength,
	PGPUInt32 *pnewobj )
{
	PGPKeyDB *		keydb = (PGPKeyDB *)kdb_id;
	PGPKeyDBObj *	obj = (PGPKeyDBObj *) obj_id;
	PGPKeyDBObj *	newobj;
	PGPError		err;

	if (pgpRPCEnabled())
		return pgpKeyDBAddObject_backRPC( context, kdb_id, obj_id, newobjs, newobjslength, pnewobj );

	pgpAssert( pgpKeyDBIsValid( keydb ) );
	pgpAssert( pgpKeyDBObjIsValid( obj ) );

	err = pgpKeyDBAddObject_internal( keydb, obj, &newobj );
	if( IsPGPError( err ) )
		return err;

	*pnewobj = (PGPUInt32)newobj;

	/* Notification is done in pgpKeyDBArray, called afterwards */
	return pgpCreateKeyArray( context, PGPPeekKeyDBObjKey(newobj), 1, NULL,
							  newobjs, newobjslength );
}


	PGPError
pgpKeyDBRemoveObject_back( PGPContextRef context, PGPUInt32 kdb_id,
	PGPUInt32 obj_id )
{
	PGPKeyDB *		keydb = (PGPKeyDB *)kdb_id;
	PGPKeyDBObj *	obj = (PGPKeyDBObj *) obj_id;
	PGPError		err = kPGPError_NoErr;
	
	if (pgpRPCEnabled())
		return pgpKeyDBRemoveObject_backRPC( context, kdb_id, obj_id );
	pgpAssert( pgpKeyDBIsValid( keydb ) );
	pgpAssert( pgpKeyDBObjIsValid( obj ) );

	if( pgpKeyDBObjIsReal( obj ) )
	{
		/*
		 * Record for notification purposes as "new" because that is the
		 * only pass-back which can handle deleted objects in Update
		 */
		(void)pgpSetPendingNotify( NULL, NULL, NULL, PGPPeekKeyDBObjKey(obj) );
		err = pgpKeyDBRemoveObject_internal( keydb, obj );
	}
	return err;
}

	PGPError
pgpCopyKeys_back( PGPContextRef context, PGPUInt32 srcid, PGPUInt32 dstid,
	PGPUInt32 *keylist, PGPSize keylistsize, PGPBoolean neednewkeylist,
	PGPUInt32 **newkeylist, PGPSize *newkeylistsize )
{
	PGPKeySetRef			keyset;
	PGPKeySetRef			newkeyset;
	PGPError				err;
	PGPKeyDB *				srckdb;
	PGPKeyDB *				dstkdb;

	if (pgpRPCEnabled())
		return pgpCopyKeys_backRPC( context, srcid, dstid, keylist,
									keylistsize, neednewkeylist, newkeylist,
									newkeylistsize );
	srckdb = (PGPKeyDB *)srcid;
	dstkdb = (PGPKeyDB *)dstid;
	pgpAssert( pgpKeyDBIsValid( srckdb ) && pgpKeyDBIsValid( dstkdb ) );
	err = pgpKeySetUnflattenFree( srckdb, keylist, keylistsize, &keyset );
	if( IsPGPError( err ) )
		return err;
	err = pgpCopyKeys_internal( keyset, dstkdb,
								(neednewkeylist?&newkeyset:NULL) );
	if( IsPGPError( err ) )
		return err;
	if( neednewkeylist )
		return pgpKeySetFlattenFree( newkeyset, newkeylist, newkeylistsize );
	/* Notification is done in pgpKeyDBArray, called afterwards */
	return kPGPError_NoErr;
}

	PGPError
pgpGlobalRandomPoolAddState_back( PGPBoolean addKeyState, PGPInt32 keyEvent,
	PGPBoolean addMouseState, PGPBoolean addSystemState )
{
	if (pgpRPCEnabled())
		return pgpGlobalRandomPoolAddState_backRPC( addKeyState, keyEvent,
											addMouseState, addSystemState );

	return pgpGlobalRandomPoolAddState_internal( addKeyState, keyEvent,
											addMouseState, addSystemState );
}

	PGPError
pgpGlobalRandomPoolGetInfo_back( PGPUInt32 *entropy, PGPUInt32 *size,
	PGPUInt32 *minEntropy, PGPBoolean *hasMinEntropy,
	PGPBoolean *hasIntelRNG )
{
	if (pgpRPCEnabled())
		return pgpGlobalRandomPoolGetInfo_backRPC( entropy, size,
									minEntropy, hasMinEntropy, hasIntelRNG );

	return pgpGlobalRandomPoolGetInfo_internal( entropy, size,
									minEntropy, hasMinEntropy, hasIntelRNG );
}


	PGPError
pgpRandomGetBytesEntropy_back( PGPContextRef context, PGPUInt32 requestBytes,
	PGPUInt32 entropyBits, PGPByte **randbuf, PGPSize *randbuflen )
{
	PGPRandomContext *rc;
	PGPByte *buf;

	if (pgpRPCEnabled())
		return pgpRandomGetBytesEntropy_backRPC( context, requestBytes,
										entropyBits, randbuf, randbuflen );

	*randbuf = NULL;
	*randbuflen = 0;
	rc = pgpContextGetX9_17RandomContext( context );
	buf = pgpContextMemAlloc( context, requestBytes, 0 );
	if( IsNull( buf ) )
		return kPGPError_OutOfMemory;
	pgpRandomGetBytesEntropy( rc, buf, requestBytes, entropyBits );
	*randbuf = buf;
	*randbuflen = requestBytes;
	return kPGPError_NoErr;
}


	PGPError
pgpRandomAddBytes_back( PGPContextRef context, PGPByte const *buf,
	PGPSize buflen )
{
	PGPRandomContext *rc;

	if (pgpRPCEnabled())
		return pgpRandomAddBytes_backRPC( context, buf, buflen );

	rc = pgpContextGetX9_17RandomContext( context );
	pgpRandomAddBytes( rc, buf, buflen );
	return kPGPError_NoErr;
}

	PGPError
pgpRandomStir_back( PGPContextRef context )
{
	PGPRandomContext *rc;

	if (pgpRPCEnabled())
		return pgpRandomStir_backRPC( context );

	rc = pgpContextGetX9_17RandomContext( context );
	pgpRandomStir( rc );
	return kPGPError_NoErr;
}

	PGPError
pgpSetRandSeedFile_back( PFLFileSpecRef randFile )
{
	if (pgpRPCEnabled())
		return pgpSetRandSeedFile_backRPC( randFile );

	return pgpSetRandSeedFile_internal( randFile );
}



/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
